/*

 MemoryBlock Class for C++
 (c) 2004 Jonathan Bettencourt / Kreative Korporation

 This is the first thing I've written for C++ outside of a Hello World program,
 so please be patient with my possibly inefficient coding.
 (Such an odd thing to start out with, something this advanced, don't you think?
 I think it's good that I'm doing this, to get a feel for memory management.)


 This basically imitates the MemoryBlock class found in RealBasic. You create
 a memoryblock of a specified size, which becomes your block of memory to do with
 as you please. You can read and write values of different data types from any
 memory location in the memoryblock. You can also change the size later, if you
 want. Memoryblock is (or at least should be) clean, handling all of the memory
 management for you and never leaving the possibility of losing track of it all.

 There are two sets of get/set functions. One set is named like the C++ data types
 and accepts and returns specific types. The other set is named like the properties
 of a RealBasic MemoryBlock and use just one data type for a general data type,
 regardless of size (int for all integers, double for all real numbers, char* for
 all strings, etc.).

 The setting of the littleEndian flag only affects the following:

	get/setLong
	get/setShort
	get/setUShort

 This class does no bounds checking. Come to think of it, I don't think the
 RealBasic version does bounds checking either.


 This code is under the MIT license.

 Permission is hereby granted, free of charge, to any person obtaining a copy of
 this software and associated documentation files (the "Software"), to deal in the
 Software without restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
 Software, and to permit persons to whom the Software is furnished to do so,
 subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies
 or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include "memoryblock.h"
#include <string>
using namespace std;

memoryblock::memoryblock(void)
{
	mydata = new char[1];
	mydata[0] = 0;
	mysize = 1;
	setDefaultEndianness();
}

memoryblock::memoryblock(int size)
{
	mydata = new char[size];
	mysize = size;
	setDefaultEndianness();
}

memoryblock::memoryblock(const memoryblock& m)
{
	mydata = new char[m.mysize];
	memcpy(mydata, m.mydata, m.mysize);
	mysize = m.mysize;
	littleEndian = m.littleEndian;
}

memoryblock::~memoryblock()
{
	clear();
}

const memoryblock& memoryblock::operator = (const memoryblock& m)
{
	clear();
	mydata = new char[m.mysize];
	memcpy(mydata, m.mydata, m.mysize);
	mysize = m.mysize;
	littleEndian = m.littleEndian;
	return *this;
}

void memoryblock::clear(void)
{
	delete [] mydata;
	mysize = 0;
}

void memoryblock::zeroOut(void)
{
	int i;
	for (i=0; i<mysize; ++i)
	{
		mydata[i] = 0;
	}
}

void memoryblock::zeroOut(int startpos)
{
	int i;
	for (i=startpos; i<mysize; ++i)
	{
		mydata[i] = 0;
	}
}

void memoryblock::zeroOut(int startpos, int endpos)
{
	int i;
	for (i=startpos; i<endpos; ++i)
	{
		mydata[i] = 0;
	}
}

void memoryblock::setDefaultEndianness(void)
{
	short int a = 0x0001;
	char * b = (char *) &a;
	littleEndian = (b[0]?true:false);
}

int memoryblock::getSize(void)
{
	return mysize;
}

void memoryblock::setSize(int nusize)
{
	char * tmp = new char[nusize];
	int amtToCpy = min(nusize, mysize);
	memcpy(tmp, mydata, amtToCpy);
	clear();
	mydata = tmp;
	mysize = nusize;
	if (amtToCpy<mysize)
	{
		zeroOut(amtToCpy);
	}
}

/*
 REALBASIC GET/SET FUNCTIONS
 These imitate the properties of MemoryBlock objects in RealBasic.
 They use a general data type (bool, int, double, char*, or string)
 for all data types, regardless of size. These are mostly compiler-
 and platform-independent.
*/

bool memoryblock::getBooleanValue(int pos)
{
	return (mydata[pos] != 0);
}

int memoryblock::getByte(int pos)
{
	return mydata[pos];
}

char * memoryblock::getCString(int pos)
{
	char * t;
	t = new char[strlen(mydata+pos)+1];
	strcpy(t, mydata+pos);
	return t;
}

string memoryblock::getCStringAsString(int pos)
{
	char * t;
	string s;
	t = new char[strlen(mydata+pos)+1];
	strcpy(t, mydata+pos);
	s = t;
	delete [] t;
	return s;
}

double memoryblock::getDoubleValue(int pos)
{
	double t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

int memoryblock::getLong(int pos)
{
	int i;
	char * t;
	t = new char[4];
	memcpy(t, mydata+pos, 4);
	if (littleEndian)
	{
		i = t[0] + (256 * t[1]) + (65536 * t[2]) + (16777216 * t[3]);
	} else {
		i = t[3] + (256 * t[2]) + (65536 * t[1]) + (16777216 * t[0]);
	}
	delete [] t;
	return i;
}

char * memoryblock::getPString(int pos)
{
	char * t;
	int sl;
	sl = mydata[pos];
	sl++;
	t = new char[sl];
	memcpy(t, mydata+pos, sl);
	return t;
}

char * memoryblock::getPStringAsCString(int pos)
{
	char * t;
	int sl;
	int i;
	sl = mydata[pos];
	t = new char[sl+1];
	t[sl] = 0;
	if (sl>0)
	{
		memcpy(t, mydata+pos+1, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	return t;
}

string memoryblock::getPStringAsString(int pos)
{
	char * t;
	string s;
	int sl;
	int i;
	sl = mydata[pos];
	t = new char[sl+1];
	t[sl] = 0;
	if (sl>0)
	{
		memcpy(t, mydata+pos+1, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	s = t;
	delete [] t;
	return s;
}

int memoryblock::getShort(int pos)
{
	int i;
	char * t;
	t = new char[2];
	memcpy(t, mydata+pos, 2);
	if (littleEndian)
	{
		i = t[0] + (256 * t[1]);
	} else {
		i = t[1] + (256 * t[0]);
	}
	delete [] t;
	return (i<32768)?i:(i-65536);
}

double memoryblock::getSingleValue(int pos)
{
	float t;
	memcpy(&t, mydata+pos, sizeof(t));
	return (double)t;
}

char * memoryblock::getStringValue(int pos, int sl)
{
	char * t;
	t = new char[sl];
	memcpy(t, mydata+pos, sl);
	return t;
}

char * memoryblock::getStringValueAsCString(int pos, int sl)
{
	char * t;
	int i;
	t = new char[sl+1];
	t[sl] = 0;
	if (sl>0)
	{
		memcpy(t, mydata+pos, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	return t;
}

string memoryblock::getStringValueAsString(int pos, int sl)
{
	char * t;
	string s;
	int i;
	t = new char[sl+1];
	t[sl] = 0;
	if (sl>0)
	{
		memcpy(t, mydata+pos, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	s = t;
	delete [] t;
	return s;
}

int memoryblock::getUShort(int pos)
{
	int i;
	char * t;
	t = new char[2];
	memcpy(t, mydata+pos, 2);
	if (littleEndian)
	{
		i = t[0] + (256 * t[1]);
	} else {
		i = t[1] + (256 * t[0]);
	}
	delete [] t;
	return i;
}

void memoryblock::setBooleanValue(int pos, bool value)
{
	mydata[pos] = value?1:0;
}

void memoryblock::setByte(int pos, int value)
{
	mydata[pos] = value;
}

void memoryblock:: setCString(int pos, char * value)
{
	strcpy(mydata+pos, value);
}

void memoryblock:: setCStringAsString(int pos, string value)
{
	strcpy(mydata+pos, value.c_str());
}

void memoryblock::setDoubleValue(int pos, double value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setLong(int pos, int value)
{
	int a = (value / 16777216) % 256;
	int b = (value / 65536) % 256;
	int c = (value / 256) % 256;
	int d = value % 256;
	if (littleEndian)
	{
		mydata[pos] = d;
		mydata[pos+1] = c;
		mydata[pos+2] = b;
		mydata[pos+3] = a;
	} else {
		mydata[pos] = a;
		mydata[pos+1] = b;
		mydata[pos+2] = c;
		mydata[pos+3] = d;
	}
}

void memoryblock::setPString(int pos, char * value)
{
	int sl;
	sl = value[0];
	memcpy(mydata+pos, value, sl+1);
}

void memoryblock::setPStringAsCString(int pos, char * value)
{
	mydata[pos] = strlen(value);
	strcpy(mydata+pos+1, value);
}

void memoryblock::setPStringAsString(int pos, string value)
{
	mydata[pos] = value.size();
	strcpy(mydata+pos+1, value.c_str());
}

void memoryblock::setShort(int pos, int value)
{
	int t = value;
	int a;
	int b;
	if (t>65536) { t-=65536; }
	if (t<65536) { t+=65536; }
	a = (t / 256) % 256;
	b = t % 256;
	if (littleEndian)
	{
		mydata[pos] = b;
		mydata[pos+1] = a;
	} else {
		mydata[pos] = a;
		mydata[pos+1] = b;
	}
}

void memoryblock::setSingleValue(int pos, double value)
{
	float f;
	f = (float)value;
	memcpy(mydata+pos, &f, sizeof(f));
}

void memoryblock::setStringValue(int pos, int sl, char * value)
{
	memcpy(mydata+pos, value, sl);
}

void memoryblock::setStringValueAsCString(int pos, int sl, char * value)
{
	int sl2;
	sl2 = min(strlen(value)+1, sl);
	memcpy(mydata+pos, value, sl2);
	if (sl2 < sl)
	{
		zeroOut(sl2, sl);
	}
}

void memoryblock::setStringValueAsString(int pos, int sl, string value)
{
	char * t;
	t = new char[value.size() + 1];
	strcpy(t, value.c_str());
	setStringValueAsCString(pos, sl, t);
	delete [] t;
}

void memoryblock::setUShort(int pos, int value)
{
	int t = value;
	int a;
	int b;
	if (t>65536) { t-=65536; }
	if (t<65536) { t+=65536; }
	a = (t / 256) % 256;
	b = t % 256;
	if (littleEndian)
	{
		mydata[pos] = b;
		mydata[pos+1] = a;
	} else {
		mydata[pos] = a;
		mydata[pos+1] = b;
	}
}

/*
 C++ GET/SET FUNCTIONS
 These use specific C++ data types. They are compiler- and platform-dependent,
 like regular C++ variables.
*/

bool memoryblock::getBool(int pos)
{
	bool t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

char memoryblock::getChar(int pos)
{
	return mydata[pos];
}

signed char memoryblock::getSignedChar(int pos)
{
	signed char tmp = mydata[pos];
	return tmp;
}

int memoryblock::getInt(int pos)
{
	int t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

short int memoryblock::getShortInt(int pos)
{
	short int t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

long int memoryblock::getLongInt(int pos)
{
	long int t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

unsigned short int memoryblock::getUnsignedShortInt(int pos)
{
	unsigned short int t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

unsigned long int memoryblock::getUnsignedLongInt(int pos)
{
	unsigned long int t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

float memoryblock::getFloat(int pos)
{
	float t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

double memoryblock::getDouble(int pos)
{
	double t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

long double memoryblock::getLongDouble(int pos)
{
	long double t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

wchar_t memoryblock::getWchar_t(int pos)
{
	wchar_t t;
	memcpy(&t, mydata+pos, sizeof(t));
	return t;
}

string memoryblock::getString(int pos)
{
	char * t;
	string s;
	t = new char[strlen(mydata+pos)+1];
	strcpy(t, mydata+pos);
	s = t;
	delete [] t;
	return s;
}

char * memoryblock::getCharStar(int pos)
{
	char * t;
	t = new char[strlen(mydata+pos)+1];
	strcpy(t, mydata+pos);
	return t;
}

void memoryblock::setBool(int pos, bool value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setChar(int pos, char value)
{
	mydata[pos] = value;
}

void memoryblock::setSignedChar(int pos, signed char value)
{
	char tmp = value;
	mydata[pos] = tmp;
}

void memoryblock::setInt(int pos, int value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setShortInt(int pos, short int value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setLongInt(int pos, long int value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setUnsignedShortInt(int pos, unsigned short int value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setUnsignedLongInt(int pos, unsigned long int value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setFloat(int pos, float value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setDouble(int pos, double value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setLongDouble(int pos, long double value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setWchar_t(int pos, wchar_t value)
{
	memcpy(mydata+pos, &value, sizeof(value));
}

void memoryblock::setString(int pos, string value)
{
	strcpy(mydata+pos, value.c_str());
}

void memoryblock::setCharStar(int pos, char * value)
{
	strcpy(mydata+pos, value);
}

const char * memoryblock::getPtr(void)
{
	/* This returns the pointer to the memoryblock's data.
	This is useful for passing to toolbox routines and stuff like that. */
	return mydata;
}